<?php

declare(strict_types=1);

namespace Gls\GlsPoland\PrestaShop\Grid\Definition\Factory;

use Gls\GlsPoland\Controller\Admin\ConsignmentController;
use Gls\GlsPoland\PrestaShop\Grid\AccessibilityChecker\DeleteConsignmentAccessibilityChecker;
use Gls\GlsPoland\PrestaShop\Grid\Action\Bulk\ModalFormSubmitBulkAction;
use Gls\GlsPoland\PrestaShop\Grid\Column\BulkActionColumn;
use Gls\GlsPoland\PrestaShop\Grid\Column\LegacyLinkColumn;
use Gls\GlsPoland\PrestaShop\Grid\Column\ParcelNumbersListColumn;
use Gls\GlsPoland\Translation\TranslatorAwareTrait;
use PrestaShop\PrestaShop\Core\Grid\Action\Bulk\BulkActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\Bulk\BulkActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\LinkGridAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\SimpleGridAction;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollection;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\ActionColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DataColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DateTimeColumn;
use PrestaShop\PrestaShop\Core\Grid\Definition\Factory\AbstractGridDefinitionFactory;
use PrestaShop\PrestaShop\Core\Grid\Filter\Filter;
use PrestaShop\PrestaShop\Core\Grid\Filter\FilterCollection;
use PrestaShop\PrestaShop\Core\Grid\Filter\FilterCollectionInterface;
use PrestaShop\PrestaShop\Core\Hook\HookDispatcherInterface;
use PrestaShopBundle\Event\Dispatcher\NullDispatcher;
use PrestaShopBundle\Form\Admin\Type\DateRangeType;
use PrestaShopBundle\Form\Admin\Type\SearchAndResetType;
use PrestaShopBundle\Security\Voter\PageVoter;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;

final class ConsignmentGridDefinitionFactory extends AbstractGridDefinitionFactory implements FilterableGridDefinitionFactoryInterface
{
    use TranslatorAwareTrait;
    use ConsignmentGridRowActionsTrait;

    public const GRID_ID = 'gls_poland_consignment';

    private $authorizationChecker;

    public function __construct(AuthorizationCheckerInterface $authorizationChecker, DeleteConsignmentAccessibilityChecker $deleteAccessibilityChecker, ?HookDispatcherInterface $hookDispatcher = null)
    {
        parent::__construct($hookDispatcher ?? new NullDispatcher());

        $this->authorizationChecker = $authorizationChecker;
        $this->deleteAccessibilityChecker = $deleteAccessibilityChecker;
    }

    protected function getId(): string
    {
        return self::GRID_ID;
    }

    public function getFilterId(): string
    {
        return $this->getId();
    }

    protected function getName(): string
    {
        return $this->trans('Consignments', [], 'Modules.Glspoland.Consignment');
    }

    protected function getColumns(): ColumnCollectionInterface
    {
        if (!class_exists(DataColumn::class)) {
            class_alias(\PrestaShop\PrestaShop\Core\Grid\Column\Type\DataColumn::class, DataColumn::class);
        }

        return (new ColumnCollection())
            ->add(
                (new BulkActionColumn('ids'))
                    ->setOptions([
                        'bulk_field' => 'id',
                        'disabled_field' => 'pickup_id',
                    ])
            )
            ->add(
                (new DataColumn('id'))
                    ->setName($this->trans('ID', [], 'Admin.Global'))
                    ->setOptions([
                        'field' => 'id',
                    ])
            )
            ->add(
                (new LegacyLinkColumn('order_reference'))
                    ->setName($this->trans('Order', [], 'Admin.Global'))
                    ->setOptions([
                        'field' => 'order_reference',
                        'route' => 'AdminOrders',
                        'route_param_name' => 'id_order',
                        'route_param_field' => 'id_order',
                        'extra_route_params' => [
                            'vieworder' => true,
                        ],
                    ])
            )
            ->add(
                (new DataColumn('reference'))
                    ->setName($this->trans('References', [], 'Modules.Glspoland.Consignment'))
                    ->setOptions([
                        'field' => 'reference',
                    ])
            )
            ->add(
                (new DataColumn('email'))
                    ->setName($this->trans('Email', [], 'Admin.Global'))
                    ->setOptions([
                        'field' => 'email',
                    ])
            )
            ->add(
                (new DataColumn('phone'))
                    ->setName($this->trans('Phone', [], 'Admin.Global'))
                    ->setOptions([
                        'field' => 'phone',
                    ])
            )
            ->add(
                (new ParcelNumbersListColumn('parcel_numbers'))
                    ->setName($this->trans('Parcel numbers', [], 'Modules.Glspoland.Consignment'))
                    ->setOptions([
                        'field' => 'parcel_numbers',
                    ])
            )
            ->add(
                (new DataColumn('pickup_id'))
                    ->setName($this->trans('Pickup ID', [], 'Modules.Glspoland.Consignment'))
                    ->setOptions([
                        'field' => 'pickup_id',
                    ])
            )
            ->add(
                (new DateTimeColumn('created_at'))
                    ->setName($this->trans('Created at', [], 'Modules.Glspoland.Consignment'))
                    ->setOptions([
                        'field' => 'created_at',
                        'format' => 'Y-m-d H:i:s',
                    ])
            )
            ->add(
                (new ActionColumn('actions'))
                    ->setName($this->trans('Actions', [], 'Admin.Global'))
                    ->setOptions([
                        'actions' => $this->getRowActions(),
                    ])
            );
    }

    protected function getFilters(): FilterCollectionInterface
    {
        return (new FilterCollection())
            ->add(
                (new Filter('id', NumberType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search ID', [], 'Admin.Global'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('id')
            )
            ->add(
                (new Filter('order_reference', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search order', [], 'Modules.Glspoland.Consignment'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('order_reference')
            )
            ->add(
                (new Filter('reference', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search by references', [], 'Modules.Glspoland.Consignment'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('reference')
            )
            ->add(
                (new Filter('email', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search email', [], 'Modules.Glspoland.Consignment'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('email')
            )
            ->add(
                (new Filter('phone', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search phone', [], 'Modules.Glspoland.Consignment'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('phone')
            )
            ->add(
                (new Filter('parcel_numbers', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search parcel number', [], 'Modules.Glspoland.Consignment'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('parcel_numbers')
            )
            ->add(
                (new Filter('pickup_id', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search pickup', [], 'Modules.Glspoland.Consignment'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('pickup_id')
            )
            ->add(
                (new Filter('created_at', DateRangeType::class))
                    ->setTypeOptions([
                        'required' => false,
                    ])
                    ->setAssociatedColumn('created_at')
            )
            ->add(
                (new Filter('actions', SearchAndResetType::class))
                    ->setTypeOptions([
                        'reset_route' => 'admin_common_reset_search_by_filter_id',
                        'reset_route_params' => [
                            'filterId' => self::GRID_ID,
                        ],
                        'redirect_route' => 'admin_gls_poland_consignments_index',
                    ])
                    ->setAssociatedColumn('actions')
            );
    }

    protected function getGridActions(): GridActionCollectionInterface
    {
        return (new GridActionCollection())
            ->add(
                (new LinkGridAction('csv_export'))
                    ->setName($this->trans('Export to CSV', [], 'Modules.Glspoland.Admin'))
                    ->setIcon('cloud_download')
                    ->setOptions([
                        'route' => 'admin_gls_poland_consignments_export',
                    ])
            )
            ->add(
                (new LinkGridAction('xls_export'))
                    ->setName($this->trans('Export to XLS', [], 'Modules.Glspoland.Admin'))
                    ->setIcon('cloud_download')
                    ->setOptions([
                        'route' => 'admin_gls_poland_consignments_export',
                        'route_params' => [
                            '_format' => 'xls',
                        ],
                    ])
            )
            ->add(
                (new LinkGridAction('xlsx_export'))
                    ->setName($this->trans('Export to XLSX', [], 'Modules.Glspoland.Admin'))
                    ->setIcon('cloud_download')
                    ->setOptions([
                        'route' => 'admin_gls_poland_consignments_export',
                        'route_params' => [
                            '_format' => 'xlsx',
                        ],
                    ])
            )
            ->add(
                (new SimpleGridAction('common_refresh_list'))
                    ->setName($this->trans('Refresh list', [], 'Admin.Advparameters.Feature'))
                    ->setIcon('refresh')
            )
            ->add(
                (new SimpleGridAction('common_show_query'))
                    ->setName($this->trans('Show SQL query', [], 'Admin.Actions'))
                    ->setIcon('code')
            )
            ->add(
                (new SimpleGridAction('common_export_sql_manager'))
                    ->setName($this->trans('Export to SQL Manager', [], 'Admin.Actions'))
                    ->setIcon('storage')
            );
    }

    protected function getBulkActions(): BulkActionCollectionInterface
    {
        $collection = new BulkActionCollection();

        if ($this->authorizationChecker->isGranted(PageVoter::UPDATE, ConsignmentController::TAB_NAME)) {
            $collection->add(
                (new ModalFormSubmitBulkAction('create_pickup'))
                    ->setName($this->trans('Create pickup', [], 'Modules.Glspoland.Consignment'))
                    ->setOptions([
                        'submit_route' => 'admin_gls_poland_consignments_create_pickup',
                        'modal_id' => 'create_pickup_modal',
                    ])
            );
        }

        return $collection;
    }
}
